<?php

namespace App\Models;


use App\Helpers\Helper;
use App\Jobs\AutomationJob;
use App\Library\Tool;
use App\Models\Traits\TrackJobs;
use Carbon\Carbon;
use DateTime;
use DateTimeZone;
use Exception;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\DB;

/**
 * @method static where(string $string, string $uid)
 * @method static create(array $array)
 * @method static whereIn(string $string, array $ids)
 * @method static find(int $int)
 */
class Automation extends Model
{
    use HasFactory, TrackJobs;


    // Automation status
    const STATUS_ACTIVE   = 'active';
    const STATUS_INACTIVE = 'inactive';

    protected $fillable = [
            'user_id',
            'name',
            'contact_list_id',
            'sending_server_id',
            'message',
            'media_url',
            'language',
            'gender',
            'sms_type',
            'status',
            'reason',
            'sender_id',
            'cache',
            'timezone',
            'data',
            'running_pid',
            'dlt_template_id',
    ];

    protected $casts = [
            'created_at' => 'datetime',
            'updated_at' => 'datetime',
    ];


    /**
     * Bootstrap any application services.
     */
    public static function boot()
    {
        parent::boot();

        // Create uid when creating list.
        static::creating(function ($item) {
            // Create new uid
            $uid = uniqid();
            while (self::where('uid', $uid)->count() > 0) {
                $uid = uniqid();
            }
            $item->uid = $uid;
        });

        // Create uid when creating automation.
        static::created(function ($item) {
            $item->updateCache();
        });
    }


    /**
     * get user
     *
     * @return BelongsTo
     *
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * get customer
     *
     * @return BelongsTo
     *
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class, 'user_id');
    }

    /**
     * get sending server
     *
     * @return BelongsTo
     *
     */
    public function sendingServer(): BelongsTo
    {
        return $this->belongsTo(SendingServer::class);
    }

    /**
     * associate with contact groups
     *
     * @return BelongsTo
     */
    public function contactList(): BelongsTo
    {
        return $this->belongsTo(ContactGroups::class);
    }

    /**
     * get reports
     *
     * @return HasMany
     */
    public function reports(): HasMany
    {
        return $this->hasMany(Reports::class, 'automation_id', 'id');
    }

    /**
     * get tracking log
     *
     * @return HasMany
     */
    public function trackingLogs(): HasMany
    {
        return $this->hasMany(TrackingLog::class, 'automation_id', 'id');
    }

    /**
     * Find item by uid.
     *
     * @return object
     */
    public static function findByUid($uid)
    {
        return self::where('uid', '=', $uid)->first();
    }

    /**
     * Frequency time unit options.
     *
     * @return array
     */
    public static function timeUnitOptions(): array
    {
        return [
                ['value' => 'minute', 'text' => 'minute'],
                ['value' => 'day', 'text' => 'day'],
                ['value' => 'week', 'text' => 'week'],
                ['value' => 'month', 'text' => 'month'],
                ['value' => 'year', 'text' => 'year'],
        ];
    }


    /**
     * Get delay or before options.
     */
    public static function getDelayBeforeOptions()
    {
        return [
                ['text' => trans_choice('locale.automations.0_day', 0), 'value' => '0 day'],
                ['text' => trans_choice('locale.automations.day', 1), 'value' => '1 day'],
                ['text' => trans_choice('locale.automations.day', 2), 'value' => '2 days'],
                ['text' => trans_choice('locale.automations.day', 3), 'value' => '3 days'],
                ['text' => trans_choice('locale.automations.day', 4), 'value' => '4 days'],
                ['text' => trans_choice('locale.automations.day', 5), 'value' => '5 days'],
                ['text' => trans_choice('locale.automations.day', 6), 'value' => '6 days'],
                ['text' => trans_choice('locale.automations.week', 1), 'value' => '1 week'],
                ['text' => trans_choice('locale.automations.week', 2), 'value' => '2 weeks'],
                ['text' => trans_choice('locale.automations.month', 1), 'value' => '1 month'],
                ['text' => trans_choice('locale.automations.month', 2), 'value' => '2 months'],
        ];
    }


    /**
     * Update Automation cached data.
     *
     * @param  null  $key
     */
    public function updateCache($key = null)
    {
        // cache indexes
        $index = [
                'DeliveredCount'       => function ($automation) {
                    return $automation->deliveredCount();
                },
                'FailedDeliveredCount' => function ($automation) {
                    return $automation->failedCount();
                },
                'NotDeliveredCount'    => function ($automation) {
                    return $automation->notDeliveredCount();
                },
                'ContactCount'         => function ($automation) {
                    return $automation->contactCount(true);
                },
                'PendingContactCount'  => function ($automation) {
                    return $automation->pendingContactCount(true);
                },
        ];

        // retrieve cached data
        $cache = json_decode($this->cache, true);
        if (is_null($cache)) {
            $cache = [];
        }

        if (is_null($key)) {
            foreach ($index as $key => $callback) {
                $cache[$key] = $callback($this);
            }
        } else {
            $callback    = $index[$key];
            $cache[$key] = $callback($this);
        }

        // write back to the DB
        $this->cache = json_encode($cache);
        $this->save();
    }

    /**
     * Retrieve Automation cached data.
     *
     * @param $key
     * @param  null  $default
     *
     * @return mixed
     */
    public function readCache($key, $default = null): mixed
    {
        $cache = json_decode($this->cache, true);
        if (is_null($cache)) {
            return $default;
        }
        if (array_key_exists($key, $cache)) {
            if (is_null($cache[$key])) {
                return $default;
            } else {
                return $cache[$key];
            }
        } else {
            return $default;
        }
    }


    public function contactCount($cache = false)
    {
        if ($cache) {
            return $this->readCache('ContactCount', 0);
        }

        return Contacts::where('group_id', $this->contactList()->id)->where('status', 'subscribe')->count();

    }

    /**
     * show delivered count
     *
     * @param  false  $cache
     *
     * @return int
     */
    public function deliveredCount(bool $cache = false): int
    {
        if ($cache) {
            return $this->readCache('DeliveredCount', 0);
        }

        return $this->reports()->where('automation_id', $this->id)->where('status', 'like', '%Delivered%')->count();
    }

    /**
     * show failed count
     *
     * @param  false  $cache
     *
     * @return int
     */
    public function failedCount(bool $cache = false): int
    {
        if ($cache) {
            return $this->readCache('FailedDeliveredCount', 0);
        }

        return $this->reports()->where('automation_id', $this->id)->where('status', 'not like', '%Delivered%')->count();
    }

    /**
     * show not delivered count
     *
     * @param  false  $cache
     *
     * @return int
     */
    public function notDeliveredCount(bool $cache = false): int
    {
        if ($cache) {
            return $this->readCache('NotDeliveredCount', 0);
        }

        return $this->reports()->where('automation_id', $this->id)->where('status', 'like', '%Sent%')->count();
    }

    /**
     * Pending Contact
     *
     * @return int
     */
    public function pendingContactCount(): int
    {
        return $this->readCache('ContactCount') - ($this->readCache('DeliveredCount') + $this->readCache('FailedDeliveredCount'));
    }

    /**
     * get sms type
     *
     * @return string
     */
    public function getSMSType(): string
    {
        $sms_type = $this->sms_type;

        if ($sms_type == 'plain') {
            return '<span class="badge bg-primary text-uppercase me-1 mb-1">'.__('locale.labels.plain').'</span>';
        }
        if ($sms_type == 'unicode') {
            return '<span class="badge bg-primary text-uppercase me-1 mb-1">'.__('locale.labels.unicode').'</span>';
        }

        if ($sms_type == 'voice') {
            return '<span class="badge bg-success text-uppercase me-1 mb-1">'.__('locale.labels.voice').'</span>';
        }

        if ($sms_type == 'mms') {
            return '<span class="badge bg-info text-uppercase me-1 mb-1">'.__('locale.labels.mms').'</span>';
        }

        if ($sms_type == 'whatsapp') {
            return '<span class="badge bg-warning text-uppercase mb-1">'.__('locale.labels.whatsapp').'</span>';
        }

        return '<span class="badge bg-danger text-uppercase mb-1">'.__('locale.labels.invalid').'</span>';
    }


    /**
     * get campaign status
     *
     * @return string
     */
    public function getStatus(): string
    {
        $status = $this->status;

        if ($status == self::STATUS_INACTIVE) {
            return '<div>
                        <span class="badge bg-warning text-uppercase mr-1 mb-1">'.__('locale.labels.paused').'</span>
                        <p class="text-muted">'.__('locale.labels.paused_at').': '.Tool::customerDateTime($this->updated_at).'</p>
                    </div>';
        }

        return '<span class="badge bg-info text-uppercase mr-1 mb-1">'.__('locale.labels.running').'</span>';
    }

    public function subscribers()
    {
        return Contacts::where('group_id', $this->contact_list_id)->select('contacts.*');
    }

    public function subscribersNotTriggeredThisYear()
    {
        $thisYear = $this->user->getCurrentTime()->format('Y');

        return $this->subscribers()->where('contacts.status', 'subscribe')->whereNotNull('contacts.birth_date')->leftJoin('tracking_logs', function ($join) {
            $join->on('tracking_logs.contact_id', 'contacts.id');
            $join->where('tracking_logs.automation_id', $this->id);
        })->where(function ($query) use ($thisYear) {
            $query->whereNull('tracking_logs.id')
                  ->orWhereRaw(sprintf('year(%s) < %s', Helper::table('tracking_logs.created_at'), $thisYear));
        });
    }

    public function getOptions()
    {
        return json_decode($this->data, true)['options'];
    }


    /**
     * @throws Exception
     */
    public function start()
    {

        $currentTime = new DateTime(null, new DateTimeZone($this->timezone));
        $triggerTime = new DateTime($this->getOptions()['at'], new DateTimeZone($this->timezone));

        if ($currentTime < $triggerTime) {
            return;
        }

        $interval = $this->getOptions()['before'];
        $today    = Carbon::now($this->timezone)->modify($interval);

        $contacts = $this->subscribersNotTriggeredThisYear()->whereIn(
                DB::raw("DATE_FORMAT(STR_TO_DATE(".Helper::table('contacts.birth_date').", '".config('custom.date_format_sql')."'), '%m-%d')"),
                [$today->format('m-d')]
        )->get();

        // Delete previous ScheduleCampaign jobs
        $this->cancelAndDeleteJobs(AutomationJob::class);

        // Schedule Job initialize
        $scheduler = (new AutomationJob($this, $contacts))->delay($triggerTime);

        // Dispatch using the method provided by TrackJobs
        // to also generate job-monitor record
        $this->dispatchWithMonitor($scheduler);
    }


    /**
     * Clear existing jobs
     *
     * @return void
     */
    public function cancelAndDeleteJobs()
    {
        JobMonitor::where('subject_name', self::class)->where('subject_id', $this->id)->delete();
    }

    public function getSubscribersWithTriggerInfo()
    {
        return $this->subscribers()
                    ->leftJoin('tracking_logs', function ($join) {
                        $join->on('tracking_logs.contact_id', 'contacts.id');
                        $join->where('tracking_logs.automation_id', $this->id);
                    })
                    ->addSelect('tracking_logs.id as auto_trigger_id')
                    ->addSelect('tracking_logs.status as status')
                    ->addSelect('tracking_logs.created_at as triggered_at');
    }

    public function sendNow(Automation $automation, Contacts $contact)
    {

    }

    /**
     * get route key by uid
     *
     * @return string
     */
    public function getRouteKeyName(): string
    {
        return 'uid';
    }


    /**
     * @return string
     */
    public function __toString(): string
    {
        return $this->name;
    }


}
