<?php

namespace App\Jobs;


use App\Library\SMSCounter;
use App\Library\Tool;
use App\Models\PlansCoverageCountries;
use App\Models\SendCampaignSMS;
use App\Models\SpamWord;
use App\Models\TrackingLog;
use App\Models\User;
use Exception;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumberUtil;

class SendAutomationMessage implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $automation;
    protected $contacts;
    protected $server;
    protected $user;
    protected $senderIds = null;

    /**
     * Create a new job instance.
     */
    public function __construct($automation, $contact, $server, $user)
    {
        $this->automation = $automation;
        $this->contacts   = $contact;
        $this->server     = $server;
        $this->user       = $user;
    }

    /**
     * pick sender id
     *
     *
     */
    public function pickSenderIds(): int|string
    {
        $selection = array_values($this->getSenderIds());
        shuffle($selection);
        while (true) {
            $element = array_pop($selection);
            if ($element) {
                return (string) $element;
            }
        }
    }

    /**
     * get sender ids
     *
     * @return array
     */
    public function getSenderIds(): array
    {

        if ( ! is_null($this->senderIds)) {
            return $this->senderIds;
        }

        $result = json_decode($this->automation->sender_id, true);

        $this->senderIds = $result;

        return $this->senderIds;
    }

    /**
     * get coverage
     *
     * @return array
     */
    public function getCoverage(): array
    {
        $data = [];
        $user = User::find($this->automation->user_id);
        if ($user) {
            $plan_coverage = PlansCoverageCountries::where('plan_id', $user->customer->activeSubscription()->plan->id)->cursor();
            foreach ($plan_coverage as $coverage) {
                $data[$coverage->country->country_code] = json_decode($coverage->options, true);
            }
        }

        return $data;

    }

    /**
     * Execute the job.
     *
     * @throws Exception
     */
    public function handle()
    {
        if ($this->batch()->cancelled()) {
            return false;
        }

        $sendMessage = new SendCampaignSMS();

        $sending_server = $this->server;
        $coverage       = $this->getCoverage();

        if (count($coverage) == 0) {
            return false;
        }

        $check_sender_id = $this->getSenderIds();

        foreach ($this->contacts as $subscriber) {

            if ($this->user->customer->getOption('sms_max') != '-1' && $this->user->sms_unit <= 0) {
                $this->batch()->cancel();

                return false;
            }

            if ($this->user->customer->overQuota()) {
                $this->batch()->cancel();

                return false;
            }

            if ($this->server->overQuota()) {
                $this->batch()->cancel();

                return false;
            }

            $cost = 0;

            if (config('app.trai_dlt')) {
                $message = $this->automation->message;
            } else {
                $renderData   = $subscriber->toArray();
                $customFields = $subscriber->custom_fields;
                foreach ($customFields as $field) {
                    $renderData[$field->tag] = $field->value;
                }
                $message = Tool::renderSMS($this->automation->message, $renderData);
            }

            if (count($check_sender_id) > 0) {
                $sender_id = $this->pickSenderIds();
            } else {
                $sender_id = null;
            }
            $phone = str_replace(['+', '(', ')', '-', ' '], '', $subscriber->phone);

            if (Tool::validatePhone($phone)) {

                try {
                    $phoneUtil         = PhoneNumberUtil::getInstance();
                    $phoneNumberObject = $phoneUtil->parse('+'.$phone);
                    $country_code      = $phoneNumberObject->getCountryCode();
                    if (is_array($coverage) && array_key_exists($country_code, $coverage)) {

                        if ($this->automation->sms_type == 'plain' || $this->automation->sms_type == 'unicode') {
                            $cost = $coverage[$country_code]['plain_sms'];
                        }

                        if ($this->automation->sms_type == 'voice') {
                            $cost = $coverage[$country_code]['voice_sms'];
                        }

                        if ($this->automation->sms_type == 'mms') {
                            $cost = $coverage[$country_code]['mms_sms'];
                        }

                        if ($this->automation->sms_type == 'whatsapp') {
                            $cost = $coverage[$country_code]['whatsapp_sms'];
                        }

                        $sms_counter  = new SMSCounter();
                        $message_data = $sms_counter->count($message);
                        $sms_count    = $message_data->messages;

                        $price = $cost * $sms_count;

                        $preparedData['user_id']   = $this->automation->user_id;
                        $preparedData['phone']     = $phone;
                        $preparedData['sender_id'] = $sender_id;
                        $preparedData['message']   = $message;
                        $preparedData['sms_type']  = $this->automation->sms_type;
                        $preparedData['cost']      = (int) $price;
                        $preparedData['status']    = null;

                    } else {

                        $sms_counter  = new SMSCounter();
                        $message_data = $sms_counter->count($message);
                        $sms_count    = $message_data->messages;

                        $price = 1 * $sms_count;

                        $preparedData['user_id']   = $this->automation->user_id;
                        $preparedData['phone']     = $phone;
                        $preparedData['sender_id'] = $sender_id;
                        $preparedData['message']   = $message;
                        $preparedData['sms_type']  = $this->automation->sms_type;
                        $preparedData['cost']      = (int) $price;
                        $preparedData['status']    = "Permission to send an SMS has not been enabled for the region indicated by the 'To' number: ".$phone;

                    }
                } catch (NumberParseException $exception) {
                    $sms_counter  = new SMSCounter();
                    $message_data = $sms_counter->count($message);
                    $sms_count    = $message_data->messages;

                    $price = 1 * $sms_count;

                    $preparedData['user_id']   = $this->automation->user_id;
                    $preparedData['phone']     = $phone;
                    $preparedData['sender_id'] = $sender_id;
                    $preparedData['message']   = $message;
                    $preparedData['sms_type']  = $this->automation->sms_type;
                    $preparedData['cost']      = (int) $price;
                    $preparedData['status']    = $exception->getMessage();

                }
            } else {
                $sms_counter  = new SMSCounter();
                $message_data = $sms_counter->count($message);
                $sms_count    = $message_data->messages;

                $price = 1 * $sms_count;

                $preparedData['user_id']   = $this->automation->user_id;
                $preparedData['phone']     = $phone;
                $preparedData['sender_id'] = $sender_id;
                $preparedData['message']   = $message;
                $preparedData['sms_type']  = $this->automation->sms_type;
                $preparedData['cost']      = (int) $price;
                $preparedData['status']    = __('locale.customer.invalid_phone_number', ['phone' => $phone]);
            }

            if ($this->user->customer->getOption('send_spam_message') == 'no') {
                $spamWords = SpamWord::all()->filter(function ($spamWord) use ($message) {
                    if (true === str_contains(strtolower($message), strtolower($spamWord->word))) {
                        return true;
                    }

                    return false;
                });

                if ($spamWords->count()) {
                    $preparedData['status'] = 'Your message contains spam words.';
                }
            }

            $preparedData['automation_id']  = $this->automation->id;
            $preparedData['sending_server'] = $sending_server;
            if (isset($this->automation->dlt_template_id)) {
                $preparedData['dlt_template_id'] = $this->automation->dlt_template_id;
            }

            $status = null;
            if ($this->automation->sms_type == 'plain' || $this->automation->sms_type == 'unicode') {
                $status = $sendMessage->sendPlainSMS($preparedData);
            }

            if ($this->automation->sms_type == 'voice') {

                $preparedData['language'] = $this->automation->language;
                $preparedData['gender']   = $this->automation->gender;

                $status = $sendMessage->sendVoiceSMS($preparedData);
            }

            if ($this->automation->sms_type == 'mms') {
                $preparedData['media_url'] = $this->automation->media_url;
                $status                    = $sendMessage->sendMMS($preparedData);
            }

            if ($this->automation->sms_type == 'whatsapp') {
                if (isset($this->automation->media_url)) {
                    $preparedData['media_url'] = $this->automation->media_url;
                }

                if (isset($this->automation->language)) {
                    $preparedData['language'] = $this->automation->language;
                }
                $status = $sendMessage->sendWhatsApp($preparedData);
            }

            if (substr_count($status, 'Delivered') == 1) {
                $this->automation->updateCache('DeliveredCount');
            } else {
                $this->automation->updateCache('FailedDeliveredCount');
            }

            $this->automation->updateCache('PendingContactCount');

            $this->track_message($status, $subscriber, $this->server);
        }

        return true;
    }


    /**
     * @throws Exception
     */
    public function track_message($response, $subscriber, $server)
    {

        $params = [
                'message_id'        => $response->id,
                'customer_id'       => $response->user_id,
                'sending_server_id' => $server->id,
                'automation_id'     => $this->automation->id,
                'contact_id'        => $subscriber->id,
                'contact_group_id'  => $subscriber->group_id,
                'status'            => $response->status,
        ];

        if ( ! isset($params['runtime_message_id'])) {
            $params['runtime_message_id'] = $response->id;
        }

        TrackingLog::create($params);

        $this->user->customer->countUsage();
        $server->countUsage();
        if (substr_count($response->status, 'Delivered') == 1) {
            if ($this->user->sms_unit != '-1') {
                $this->user->countSMSUnit($response->cost);
            }
        }
    }

}
